#ifndef __CVstTempo__
#define __CVstTempo__

#include <Exceptions/CException.hpp>
#include <audioeffectx.h>
#include <Basics/CCountedObject.hpp>
#include <Music/CTimeSignature.hpp>

//	===========================================================================

using Exponent::Exceptions::CException;
using Exponent::Basics::CCountedObject;
using Exponent::Music::CTimeSignature;

//	===========================================================================

namespace Exponent
{
	namespace Vst
	{
		/**
		 * @class CVstTempo CVstTempo.hpp
		 * @brief Gets and store info about the Vst Tempo information
		 *
		 * @date 15/02/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Replaced virtual functions with non virtual versions
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 * Please note that VST is copyright Steinberg Media GmBh. No challenge is made to any of their trademarks
		 * To use this file, you require a copy of the VST SDK, available from www.steinberg.net for free
		 *
		 * $Id: CVstTempo.hpp,v 1.3 2007/02/08 21:08:23 paul Exp $
		 */
		class CVstTempo : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

			/**
			 * @struct SSongInformation CVstTempo.hpp
			 * @brief Stores all the information about the song for the current process buffer
			 */
			struct SSongInformation
			{
				bool m_tempoIsValid;					/**< Is the tempo valid */
				bool m_timeSignatureIsValid;			/**< Is the time signature valid */
				bool m_ppqPositionIsValid;				/**< Is the ppq position valid */
				bool m_barPositionIsValid;				/**< Is the bar position valid */
				bool m_samplesToNextBarIsValid;			/**< Is the samples to next bar variable valid */
				bool m_samplesPerBeatIsValid;			/**< Is the samples per beat variable valid */

				bool m_transportChanged;				/**< Did the transport controls change this buffer */
				bool m_transportIsPlaying;				/**< Is the transport active */

				CTimeSignature m_timeSignature;			/**< Current time signature */
				double m_bpm;							/**< Current bpm */
				double m_ppqPosition;					/**< Current ppq position */
				double m_barPosition;					/**< Current bar position */
				double m_samplePosition;				/**< Current sample position */

				long m_samplesToNextBar;				/**< Number of samples to the start of the next bar */
				long m_samplesPerBeat;					/**< Number of samples per musical beat */

				/**
				 * Intialise all values
				 */
				void initialise()
				{
					m_tempoIsValid			  = false;
					m_timeSignatureIsValid	  = false;
					m_ppqPositionIsValid	  = false;
					m_barPositionIsValid	  = false;
					m_transportChanged		  = false;
					m_transportIsPlaying	  = false;
					m_samplesToNextBarIsValid = false;
					m_samplesPerBeatIsValid	  = false;
					m_bpm					  = 0.0;
					m_ppqPosition		      = 0.0;
					m_barPosition		      = 0.0;
					m_samplePosition	      = 0.0;
					m_samplesToNextBar        = 0;
					m_samplesPerBeat          = 0;
					m_timeSignature.setTimeSignature(4, 4);
				}
			};

//	===========================================================================

			const static long CVST_TEMPO_DEFAULT_VST_FLAGS = kVstTempoValid | kVstTransportChanged | kVstBarsValid | kVstPpqPosValid | kVstTimeSigValid;	/**< Everything valid */
			const static long CVST_TEMPO_TEMPO_ONLY_FLAG   = kVstTempoValid;																				/**< Only tempo valid */

//	===========================================================================

			/**
			 * Construction
			 */
			CVstTempo();

			/**
			 * Destruction
			 */
			virtual ~CVstTempo();

			/**
			 * Initialise from AudioEffectX
			 * @param theEffect The Effect to intialise from
			 */
			void initialiseFromAudioEffectX(AudioEffectX *theEffect);

//	===========================================================================

			/**
			 * Get tempo information for this block buffer
			 * @retval bool True if correctly acquired tempo info, false otherwise
			 */
			bool getTempoInformation(const long flags = CVST_TEMPO_DEFAULT_VST_FLAGS);

			/**
			 * Is the tempo valid
			 * @retval bool True if the tempo is valid for this process block, false on error or if not available
			 */
			bool lastTempoIsValid() const { return m_lastTempoValid; }

			/**
			 * Get the tempo information
			 * @retval SSongInformation* Current process block song info
			 */
			SSongInformation *getSongInformation() const { return m_songInformation; }

//	===========================================================================

		protected:

//	===========================================================================

			AudioEffectX *m_theEffect;				/**< The effect to connect to for tempo info */
			bool m_lastTempoValid;					/**< Is tempo information currently valid */
			SSongInformation *m_songInformation;	/**< Information about the song */
		};
	}
}
#endif	// End of CVstTempo.hpp